/**
 * \file           CDeviceDriver.cpp
 * \brief          Source for CDeviceDriver base class
 * \author         RJen
 * \date           03/21/2012
 **/

#include "nutypedefs.h"
#include "CDeviceDriver.h"
#include "os.h"
#include "memorymanager.h"
#include <string.h>

/**
 *
 * \author      RJen
 * \brief       CDeviceDriver class first phase constructor
 * \detail      Constructor
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     Must be used as a base class
 * \note        none
 *
 */
CDeviceDriver::CDeviceDriver(void)
{
  m_hDriverLock = NULL;
}

/**
 *
 * \author      RJen
 * \brief       CDeviceDriver class destructor
 * \detail      Destroys all resources used by the class
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     none
 * \note        none
 *
 */
CDeviceDriver::~CDeviceDriver(void)
{
  if (m_hDriverLock)
    OsDeleteLock(m_hDriverLock);
}

/**
 *
 * \author      RJen
 * \brief       Method for initializing the class
 * \detail      Allocate and initalize all dynamic resources (second phase constructor)
 * \date        04/02/2012
 * \param       none
 * \return      bool
 * \retval      true - if initialization is successful, false - if not
 * \warning     none
 * \note        Must be checked by the derived class and if it fails, it should be destroyed
 *
 */
bool CDeviceDriver::BaseInitialize()
{
  OsCreateLock(&m_hDriverLock);
  return (m_hDriverLock);
}

/**
 *
 * \author      RJen
 * \brief       Locks the driver
 * \detail      Locks the driver
 * \date        04/02/2012
 * \param       none
 * \return      bool
 * \retval      true - if the lock is successful, false - if not
 * \warning     none
 * \note        Will block indefinitely until access to the lock is granted
 *
 */
bool CDeviceDriver::LockDriver(void)
{
  if (m_hDriverLock != NULL)
    return (OsLock(m_hDriverLock) == 0);
  else
    return false;
}

/**
 *
 * \author      RJen
 * \brief       Unlocks the driver
 * \detail      Unlocks the driver
 * \date        04/02/2012
 * \param       none
 * \return      bool
 * \retval      true - if the unlock is successful, false - if not
 * \warning     none
 * \note        Will block indefinitely until access to the lock is granted
 *
 */
bool CDeviceDriver::UnlockDriver(void)
{
  if (m_hDriverLock != NULL)
    return (OsUnlock(m_hDriverLock) == 0);
  else
    return false;
}

/**
 *
 * \author      RJen
 * \brief       CDeviceDriverIsr class first phase constructor
 * \detail      Constructor
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     Must be used as a base class
 * \note        none
 *
 */
CDeviceDriverIsr::CDeviceDriverIsr(void)
{
  m_hIsrEvent = NULL;
  m_hIsrHandlerTask = NULL;
}

/**
 *
 * \author      RJen
 * \brief       CDeviceDriverIsr class destructor
 * \detail      Destroys all resources used by the class
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     none
 * \note        none
 *
 */
CDeviceDriverIsr::~CDeviceDriverIsr(void)
{
  if (m_hIsrEvent)
    OsDeleteEvent(m_hIsrEvent);
  if (m_hIsrHandlerTask)
    OsDeleteAdvTask(m_hIsrHandlerTask);
}

/**
 *
 * \author      RJen
 * \brief       Method for initializing the class
 * \detail      Allocate and initalize all dynamic resources (second phase constructor)
 * \date        04/02/2012
 * \param       none
 * \return      bool
 * \retval      true - if initialization is successful, false - if not
 * \warning     none
 * \note        Must be checked by the derived class and if it fails, it should be destroyed
 *
 */
bool CDeviceDriverIsr::BaseInitialize()
{
  OsCreateLock(&m_hDriverLock);
  OsCreateEvent(&m_hIsrEvent);
  return (m_hDriverLock && m_hIsrEvent);
}

/**
 *
 * \author      RJen
 * \brief       Signal the event that the ISR handler task is waiting for
 * \detail      Signal the event that the ISR handler task is waiting for
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     none
 * \note        none
 *
 */
void CDeviceDriverIsr::SignalIsrEvent(void)
{
  if (m_hIsrEvent)
    OsSignalEvent(m_hIsrEvent);
}

/**
 *
 * \author      RJen
 * \brief       Configure the ISR handler task
 * \detail      Configure the ISR handler task
 * \date        04/02/2012
 * \param       StackSize - stack size of the task
 *              Priority - task priority
 *              pTaskName - task name
 * \return      bool
 * \retval      true - if task is created successfully, false - if not
 * \warning     none
 * \note        Must be called only after the derived class is initialized
 *
 */
bool CDeviceDriverIsr::ConfigureIsrHandlerTask(UINT16 StackSize, UINT32 Priority, const signed char* pTaskName)
{
  m_hIsrHandlerTask = OsCreateNamedAdvTask(IsrHandlerTaskWrapper, StackSize, (UINT32*)this, Priority, pTaskName); 
  return m_hIsrHandlerTask;
}
       
/**
 *
 * \author      RJen
 * \brief       C wrapper for ISR handler task
 * \detail      Allows tasks to be created using c++ methods
 * \date        04/02/2012
 * \param       Param - instance of the class
 * \return      none
 * \retval      none
 * \warning     none
 * \note        none
 *
 */
void CDeviceDriverIsr::IsrHandlerTaskWrapper(UINT32 Param)
{
  ((CDeviceDriverIsr*)Param)->IsrHandlerTask();
}

/**
 *
 * \author      RJen
 * \brief       ISR event handler task
 * \detail      Waits for ISR events and then call the derived handler method
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     none
 * \note        none
 *
 */
void CDeviceDriverIsr::IsrHandlerTask(void)
{
  while (true)
  {
    // Wait for event signaled by ISR and then call the handler
    OsWaitEvent(m_hIsrEvent,OS_WAIT_FOREVER);
    IsrHandler();
  }
}

/**
 *
 * \author      RJen
 * \brief       CDeviceDriverIsrInit class first phase constructor
 * \detail      Constructor
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     Must be used as a base class
 * \note        none
 *
 */
CDeviceDriverIsrInit::CDeviceDriverIsrInit(void)
{
  m_Initialized = false;
  m_hInitializerTask = NULL;
}

/**
 *
 * \author      RJen
 * \brief       CDeviceDriverIsrInit class destructor
 * \detail      Destroys all resources used by the class
 * \date        04/02/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     none
 * \note        none
 *
 */
CDeviceDriverIsrInit::~CDeviceDriverIsrInit(void)
{
  if (m_hInitializerTask)
    OsDeleteAdvTask(m_hIsrHandlerTask);
}

/**
 *
 * \author      RJen
 * \brief       Configure the initializer task
 * \detail      Configure the initializer task
 * \date        04/02/2012
 * \param       StackSize - stack size of the task
 *              Priority - task priority
 *              pTaskName - task name
 * \return      bool
 * \retval      true - if task is created successfully, false - if not
 * \warning     none
 * \note        Must be called only after the derived class is initialized
 *
 */
bool CDeviceDriverIsrInit::ConfigureInitializerTask(UINT16 StackSize, UINT32 Priority, const signed char* pTaskName)
{
  m_hInitializerTask = OsCreateNamedAdvTask(InitializerTaskWrapper, StackSize, (UINT32*)this, Priority, pTaskName); 
  return m_hInitializerTask;
}
                                
/**
  *
  * \author      RJen
  * \brief       C wrapper for initializer task
  * \detail      Also deletes task after it runs
  * \date        09/09/2011
  * \param       Param - instance of the class
  * \return      none
  * \retval      none
  * \warning     none
  * \note        none
  *
  */
void CDeviceDriverIsrInit::InitializerTaskWrapper(UINT32 Param)
{  
  CDeviceDriverIsrInit* pCurrentDriver = (CDeviceDriverIsrInit*)Param;
  
  // Call derived initializer method
  pCurrentDriver->InitializerTask();
  
  // Delete task before it returns
  OsDeleteAdvTask(pCurrentDriver->m_hInitializerTask);
  pCurrentDriver->m_hInitializerTask = NULL;
}

/**
 *
 * \author      RJen
 * \brief       CDeviceDriverQueueTask class first phase constructor
 * \detail      Constructor
 * \date        04/30/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     Must be used as a base class
 * \note        none
 *
 */
CDeviceDriverQueueTask::CDeviceDriverQueueTask(void)
{
  m_QueueSize = 0;
  m_hQueue = NULL;
  m_hDequeueTask = NULL;
}

/**
 *
 * \author      RJen
 * \brief       CDeviceDriverQueueTask class destructor
 * \detail      Destroys all resources used by the class
 * \date        04/30/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     none
 * \note        none
 *
 */
CDeviceDriverQueueTask::~CDeviceDriverQueueTask(void)
{
  if (m_hQueue)
    OsQueueDelete(m_hQueue);
  
  if (m_hDequeueTask)
    OsDeleteAdvTask(m_hDequeueTask);
}

/**
 *
 * \author      RJen
 * \brief       Initialize all the resources used by the class
 * \detail      Initialize all the resources used by the class
 * \date        04/30/2012
 * \param       QueueSize - number of items the queue can hold
 *              StackSize - stack size of the dequeue task
 *              Priority - task priority
 *              pTaskName - task name
 * \return      bool
 * \retval      true - if initialzation is successful, false - if not
 * \warning     none
 * \note        none
 *
 */
bool CDeviceDriverQueueTask::BaseInitialize(UINT32 QueueSize, UINT16 StackSize, UINT32 Priority, char* pTaskName)
{
  OsCreateLock(&m_hDriverLock); // Create lock
  m_QueueSize = QueueSize;
  m_hQueue = OsQueueCreate(m_QueueSize, sizeof(QueueData)); // Create queue
  
  // Create the task that waits on the queue
  if (m_hQueue != NULL)
    m_hDequeueTask = OsCreateNamedAdvTask(DequeueTaskWrapper, StackSize, (UINT32*)this, Priority, (signed char*)pTaskName); 
  
  return (m_hQueue && m_hDequeueTask);
}

/**
 *
 * \author      RJen
 * \brief       C wrapper for DequeueTask()
 * \detail      Allows task creation using class methods
 * \date        04/30/2012
 * \param       Param - instance of the class
 * \return      bool
 * \retval      true - if initialzation is successful, false - if not
 * \warning     none
 * \note        none
 *
 */
void CDeviceDriverQueueTask::DequeueTaskWrapper(UINT32 Param)
{  
  ((CDeviceDriverQueueTask*)Param)->DequeueTask();
}

/**
 *
 * \author      RJen
 * \brief       Task that waits on a queue item
 * \detail      Waits for a queue item and then passes it up to the derived class
 * \date        04/30/2012
 * \param       none
 * \return      none
 * \retval      none
 * \warning     Must never return; derived class must handle the memory cleanup
 * \note        none
 *
 */
void CDeviceDriverQueueTask::DequeueTask(void)
{  
  while(true)
  {
    QueueData QueueData = {NULL, 0, false, false};
    
    // Get item from queue, wait forever
    OsDequeueWait(m_hQueue, &QueueData, OS_WAIT_FOREVER, 0);
    
    // Call the derived queue handler method
    QueueHandler(&QueueData); 
  }
}

/**
 *
 * \author      RJen
 * \brief       Enqueue data by making a copy first
 * \detail      Enqueue data by making a copy first
 * \date        04/30/2012
 * \param       pData - pointer to data to queue
 *              DataSize - size of the data in bytes
 * \return      bool
 * \retval      true - if data is successfully queued, false - if not
 * \warning     none
 * \note        none
 *
 */
bool CDeviceDriverQueueTask::EnqueueCopy(UINT8* pData, UINT16 DataSize)
{  
  QueueData DataToQueue;
         
  // Create buffer for storing the data to be queued
  if ((!MemMgr) || ((DataToQueue.pData = MemMgr->GetBlock(DataSize)) == NULL))
    return false;
  
  // Set the data info and copy it onto the new buffer
  DataToQueue.DataSize = DataSize;
  DataToQueue.Copied = true;
  DataToQueue.Free = true;
  memcpy(DataToQueue.pData, pData, DataSize);
  
  // Queue the item and then check the status
  if (OsQueue(m_hQueue, &DataToQueue, 0) == -1)
  {
    // Queue failed, free the buffer and then return failure
    MemMgr->FreeBlock(DataToQueue.pData);
    return false; 
  }
  
  return true;
}

/**
 *
 * \author      RJen
 * \brief       Enqueue data by reference
 * \detail      Enqueue data by reference
 * \date        04/30/2012
 * \param       pData - pointer to data to queue
 *              DataSize - size of the data in bytes
 *              Free - if the memory has to be freed after dequeue
 * \return      bool
 * \retval      true - if data is successfully queued, false - if not
 * \warning     none
 * \note        none
 *
 */
bool CDeviceDriverQueueTask::Enqueue(UINT8* pData, UINT16 DataSize, bool Free)
{  
  QueueData DataToQueue;
        
  // Set the data pointer and info
  DataToQueue.pData = pData;
  DataToQueue.DataSize = DataSize;
  DataToQueue.Copied = false;
  DataToQueue.Free = Free;

  // Queue the item and then check the status
  if (OsQueue(m_hQueue, &DataToQueue, 0) == -1)
    return false; 
  else
    return true;
}